require(here)
source(here::here("3_Functions", "survival", "other_cause_mortality.R"))
source(here::here("3_Functions", "survival", "Survival_functions.R"))

## Testing Darren's functions ----

testthat::test_that("f_surv_getgpopOSFast works", {
  
  R_table_mort_lifeTable <- data.table(
    age..x. = 0:100,
    qx..males. = c(0.004224,0.000229,0.000127,0.000102,8.6e-05,7.4e-05,8.5e-05,6.7e-05,6.9e-05,6e-05,7.8e-05,7.7e-05,0.000102,0.000116,0.000129,0.000172,0.000205,0.000311,0.000402,0.000454,0.000525,0.000507,0.000497,0.000524,0.000556,0.000601,0.000607,0.000629,0.000681,0.000728,0.000771,0.000835,0.000858,0.000957,0.000989,0.0011,0.001155,0.001351,0.001317,0.001457,0.001606,0.0017,0.001848,0.002015,0.00221,0.002467,0.002646,0.002743,0.00296,0.003297,0.003577,0.003821,0.004075,0.004402,0.00472,0.005046,0.005593,0.00606,0.006695,0.007239,0.007912,0.008636,0.009601,0.010552,0.011171,0.01246,0.013853,0.014782,0.016348,0.017949,0.019238,0.020795,0.022782,0.025799,0.028776,0.0323,0.03573,0.040136,0.045188,0.050259,0.056143,0.06203,0.069197,0.077486,0.087296,0.097476,0.110221,0.122808,0.137654,0.154513,0.163489,0.183325,0.200869,0.222981,0.244606,0.268951,0.290494,0.31405,0.335507,0.369942,0.390506),
    qx..females. = c(0.003503,0.000214,0.000114,9.5e-05,6.4e-05,7.4e-05,7.1e-05,5.5e-05,5.8e-05,5.1e-05,6.6e-05,5.5e-05,5.7e-05,8.7e-05,9.6e-05,0.000113,0.000131,0.000158,0.000218,0.000212,0.000187,0.000211,0.000245,0.000215,0.000223,0.00026,0.000257,0.00031,0.000314,0.000338,0.000387,0.000394,0.000468,0.000493,0.000585,0.000592,0.000678,0.000761,0.000792,0.000868,0.000924,0.001004,0.001102,0.001239,0.001345,0.001484,0.001625,0.001744,0.001966,0.002051,0.002234,0.002452,0.002581,0.002764,0.002964,0.003283,0.003637,0.003928,0.004367,0.004707,0.005247,0.005636,0.006451,0.006818,0.007379,0.008113,0.00877,0.009554,0.010602,0.011458,0.012895,0.013637,0.015499,0.017289,0.019688,0.021766,0.024397,0.027918,0.03142,0.035713,0.039611,0.045127,0.050197,0.057155,0.065142,0.073748,0.08452,0.095918,0.107798,0.121609,0.136466,0.153438,0.171026,0.189568,0.20787,0.230227,0.253171,0.277939,0.299649,0.31991,0.350742)
  )
  t_yr <- seq(0, 40, by = 1/12)
  
  start_age <- 64.4
  p_male <- 0.709
  target <- c(1,0.999157028,0.998314787,0.997473277,0.996632497,0.995792447,0.994953126,0.994114533,0.993276668,0.992345274,0.99141478,0.990485185,0.98955649,0.988628693,0.987701794,0.98677579,0.985850682,0.984926469,0.984003149,0.983080723,0.982159188,0.981141039,0.980123984,0.97910802,0.978093146,0.977079362,0.976066665,0.975055055,0.97404453,0.97303509,0.972026733,0.971019457,0.970013263,0.968935629,0.967859231,0.966784069,0.965710139,0.964637442,0.963565975,0.962495737,0.961426726,0.960358942,0.959292382,0.958227045,0.957162931,0.955985793,0.954810149,0.953635998,0.952463337,0.951292165,0.95012248,0.948954279,0.947787562,0.946622325,0.945458567,0.944296286,0.94313548,0.941866018,0.940598324,0.939332396,0.93806823,0.936805824,0.935545176,0.934286283,0.933029143,0.931773753,0.930520111,0.929268213,0.928018059,0.92666497,0.925313909,0.923964874,0.922617862,0.921272868,0.919929891,0.918588928,0.917249974,0.915913027,0.914578084,0.913245142,0.911914198,0.910483657,0.909055431,0.907629514,0.906205905,0.904784598,0.90336559,0.901948877,0.900534456,0.899122322,0.897712472,0.896304902,0.894899608,0.893348523,0.891800199,0.890254629,0.88871181,0.887171735,0.885634401,0.884099803,0.882567934,0.88103879,0.879512367,0.877988659,0.876467662,0.874751932,0.873039657,0.87133083,0.869625445,0.867923493,0.866224968,0.864529863,0.862838171,0.861149885,0.859464997,0.857783502,0.856105391,0.854226089,0.852351022,0.850480178,0.84861355,0.846751126,0.844892897,0.843038854,0.841188986,0.839343285,0.837501741,0.835664343,0.833831084,0.831782012,0.82973812,0.827699393,0.825665818,0.823637381,0.821614071,0.819595872,0.817582772,0.815574758,0.813571817,0.811573936,0.8095811,0.807372136,0.805169362,0.80297276,0.800782313,0.798598002,0.796419811,0.794247722,0.792081717,0.789921779,0.78776789,0.785620034,0.783478193,0.781063694,0.77865682,0.776257548,0.773865854,0.771481712,0.769105098,0.766735988,0.764374358,0.762020185,0.759673442,0.757334108,0.755002158,0.752379395,0.749765972,0.747161856,0.744567014,0.74198141,0.739405012,0.736837786,0.734279699,0.731730717,0.729190809,0.726659939,0.724138077,0.721321818,0.718516761,0.715722861,0.712940073,0.71016835,0.70740765,0.704657926,0.701919135,0.699191232,0.696474173,0.693767914,0.691072412,0.688070867,0.685082671,0.682107762,0.679146082,0.676197569,0.673262165,0.670339809,0.667430443,0.664534007,0.661650444,0.658779694,0.6559217,0.652744721,0.649583445,0.646437793,0.643307686,0.640193046,0.637093795,0.634009856,0.630941153,0.627887607,0.624849144,0.621825688,0.618817163,0.615469389,0.612140108,0.608829217,0.605536612,0.602262191,0.59900585,0.59576749,0.592547008,0.589344304,0.586159278,0.582991832,0.579841865,0.576304927,0.572789983,0.569296893,0.565825521,0.562375728,0.558947378,0.555540335,0.552154466,0.548789636,0.545445712,0.542122563,0.538820056,0.535093949,0.531394082,0.52772027,0.524072324,0.520450061,0.516853296,0.513281848,0.509735534,0.506214176,0.502717596,0.499245615,0.495798058,0.491942352,0.488117145,0.484322192,0.480557249,0.476822077,0.473116437,0.46944009,0.465792804,0.462174343,0.458584477,0.455022977,0.451489615,0.447486304,0.443519056,0.439587543,0.435691437,0.431830416,0.428004159,0.424212349,0.420454671,0.416730814,0.413040469,0.40938333,0.405759094,0.401710667,0.397703209,0.3937363,0.389809525,0.385922473,0.382074736,0.378265912,0.374495603,0.370763413,0.367068954,0.363411837,0.359791681,0.355742965,0.351740462,0.347783637,0.343871964,0.34000492,0.336181989,0.332402663,0.328666436,0.324972812,0.321321297,0.317711407,0.31414266,0.310143514,0.306196,0.302299444,0.29845318,0.29465655,0.290908905,0.287209605,0.283558019,0.279953524,0.276395503,0.272883351,0.269416469,0.265704268,0.262043649,0.258433889,0.254874277,0.251364109,0.247902695,0.244489351,0.241123405,0.237804193,0.234531061,0.231303365,0.228120468,0.224562224,0.221059951,0.217612762,0.214219785,0.21088016,0.207593041,0.204357597,0.201173008,0.198038468,0.194953186,0.191916381,0.188927284,0.185653275,0.182436409,0.179275683,0.17617011,0.173118722,0.170120566,0.167174707,0.164280228,0.161436225,0.158641813,0.155896122,0.153198295,0.150223338,0.147306588,0.144446898,0.141643145,0.138894225,0.136199059,0.133556589,0.130965776,0.128425603,0.125935074,0.123493211,0.121099057,0.118494674,0.115946742,0.113454028,0.111015327,0.108629459,0.106295272,0.104011637,0.101777451,0.099591636,0.097453137,0.095360921,0.09331398,0.091075724,0.088891557,0.086760163,0.084680258,0.082650589,0.080669934,0.078737101,0.076850927,0.075010278,0.073214047,0.071461154,0.069750546,0.067910844,0.066119961,0.064376596,0.062679479,0.061027377,0.059419089,0.057853445,0.056329308,0.054845572,0.053401159,0.05199502,0.050626136,0.049155082,0.047726988,0.046340594,0.044994677,0.043688048,0.042419556,0.041188081,0.039992539,0.038831875,0.037705067,0.036611122,0.035549078,0.034428369,0.033343151,0.032292293,0.031274705,0.030289326,0.029335135,0.028411139,0.027516378,0.026649923,0.025810876,0.024998365,0.024211546,0.023367992,0.022554056,0.021768691,0.021010886,0.020279667,0.019574094,0.018893261,0.018236293,0.017602348,0.016990613,0.016400304,0.015830664,0.015230448,0.014653091,0.014097718,0.013563488,0.013049593,0.012555256,0.012079729,0.011622293,0.011182256,0.010758955,0.010351749,0.009960025,0.00958319,0.009220676,0.008871938,0.008536448,0.008213701,0.007903212,0.007604512,0.007317152,0.0070407,0.006774739,0.00651887,0.006272709,0.006035884,0.005808041,0.005588837,0.005377943,0.005175043,0.004979832,0.004792018,0.00461132,0.004437466,0.004270195,0.004109259,0.003954415,0.003805432,0.003662087,0.003524166,0.003391463,0.003263779,0.003140924,0.003022714,0.002908973,0.002799531,0.002694225,0.002592898,0.002495399,0.002401582,0.002311308,0.002224443,0.002140857)
  
  output <- f_surv_getgpopOSFast(p_male, start_age, t_yr, R_table_mort_lifeTable)
  
  testthat::expect_equal(target, output)
  
})


## Testing Tristan's functions ----

testthat::test_that("a single survival curve is adjusted", {
  
  v_x_lifetable <- seq(0, 100, by = 1)
  v_h_lifetable <- exp(0.1 * v_x_lifetable - 10)
  v_q_lifetable <- 1 - exp(-v_h_lifetable)
  
  v_t_os <- seq(0, 15, by = 1/52)
  # h_{OS}(t) = 0.16 * (t/5) ^ (-0.2)
  v_p_os <- pweibull(v_t_os, shape = 0.8, scale = 5, lower.tail = FALSE)
  
  # For a 60-year-old, the hazard from the OS curve exceeds the general
  # mortality hazard for the full 15 years, so adjustment should have no effect
  x <- adjust_single_survival(60, 1/52, v_t_os, v_p_os, v_x_lifetable, v_q_lifetable)
  testthat::expect_equal(x$s_adjusted, v_p_os)
  
  # For a 70-year-old, the hazard from general mortality exceeds the hazard from
  # the OS curve at a particular point. By graphical inspection this should be
  # when t = 11, so we should get a warning and survival should be adjusted
  # downwards.
  testthat::expect_warning(
    x <- adjust_single_survival(70, 1/52, v_t_os, v_p_os, v_x_lifetable, v_q_lifetable),
    "Mortality rate from life table exceeds extrapolated mortality at time 11"
  )
  testthat::expect_equal(x$s_adjusted[1:(11 * 52)], v_p_os[1:(11 * 52)])
  testthat::expect_lt(x$s_adjusted[11 * 52 + 1], v_p_os[11 * 52 + 1])
  testthat::expect_equal(x$q_adjusted[11 * 52 + 1], 1 - exp(-v_h_lifetable[v_x_lifetable == 81] / 52))
  
})


testthat::test_that("the wrapper function works with aggregate patient characteristics", {
  
  # Create i and p in the global environment with the form we want, but make
  # sure we don't clobber the real things if they are already loaded
  old_i <- if ("i" %in% names(globalenv())) get("i", envir = globalenv()) else NULL
  old_p <- if ("p" %in% names(globalenv())) get("p", envir = globalenv()) else NULL
  
  assign("i", list(R_table_mort_lifeTable = data.frame(
    V1 = 0:100,
    V2 = c(0.004233,0.000229,0.000127,0.000102,0.000086,0.000074,0.000085,0.000067,0.000069,0.00006,0.000078,0.000077,0.000102,0.000116,0.000129,0.000172,0.000205,0.000311,0.000402,0.000454,0.000525,0.000507,0.000498,0.000524,0.000556,0.000601,0.000607,0.00063,0.000681,0.000728,0.000771,0.000835,0.000859,0.000958,0.00099,0.0011,0.001155,0.001352,0.001317,0.001458,0.001607,0.001701,0.00185,0.002017,0.002212,0.00247,0.00265,0.002746,0.002964,0.003303,0.003584,0.003828,0.004084,0.004411,0.004731,0.005059,0.005609,0.006078,0.006718,0.007265,0.007944,0.008673,0.009648,0.010608,0.011234,0.012538,0.01395,0.014892,0.016483,0.018111,0.019424,0.021013,0.023044,0.026136,0.029196,0.03283,0.03638,0.040958,0.046233,0.051555,0.057764,0.064015,0.071677,0.080609,0.09128,0.10247,0.11665,0.130842,0.147828,0.16745,0.178043,0.201824,0.223296,0.250961,0.27869,0.310738,0.339857,0.372549,0.403135,0.453901,0.485254),
    V3 = c(0.003503,0.000214,0.000114,0.000095,0.000064,0.000074,0.000071,0.000055,0.000058,0.000051,0.000066,0.000055,0.000057,0.000087,0.000096,0.000113,0.000131,0.000158,0.000218,0.000212,0.000187,0.000211,0.000245,0.000215,0.000223,0.00026,0.000257,0.00031,0.000314,0.000338,0.000387,0.000394,0.000468,0.000493,0.000585,0.000592,0.000678,0.000761,0.000792,0.000868,0.000924,0.001004,0.001102,0.001239,0.001345,0.001484,0.001625,0.001744,0.001966,0.002051,0.002234,0.002452,0.002581,0.002764,0.002964,0.003283,0.003637,0.003928,0.004367,0.004707,0.005247,0.005636,0.006451,0.006818,0.007379,0.008113,0.00877,0.009554,0.010602,0.011458,0.012895,0.013637,0.015499,0.017289,0.019688,0.021766,0.024397,0.027918,0.03142,0.035713,0.039611,0.045127,0.050197,0.057155,0.065142,0.073748,0.08452,0.095918,0.107798,0.121609,0.136466,0.153438,0.171026,0.189568,0.20787,0.230227,0.253171,0.277939,0.299649,0.31991,0.350742)
  )), envir = globalenv())
  assign("p", list(basic = list(cl_y = 1/52)), envir = globalenv())
  
  # Create a survivor function to use
  t_os <- (0:208) * p$basic$cl_y
  s_os <- exp(-0.1 * t_os ^ 0.5)
  
  # Test using mean age and proportion male
  testthat::expect_warning(
    adjusted <- adjust_survival(sex = 0.55, age = 75.2, survivor = cbind(t_os, s_os), .warn = TRUE)
  )
  
  # The values below were calculated in Excel
  target <- data.frame(
    t = t_os,
    q_genmort  = c(0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000543354,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000605518,0.000687121,0.000687116,0.000687109,0.000687103,0.000687097,0.00068709,0.000687083,0.000687076,0.000687068,0.00068706,0.000687053,0.000687044,0.000687036,0.000687028,0.000687019,0.00068701,0.000687001,0.000686992,0.000686982,0.000686973,0.000686963,0.000686953,0.000686942,0.000686932,0.000686921,0.000686911,0.0006869,0.000686889,0.000686877,0.000686866,0.000686854,0.000686843,0.000686831,0.000686819,0.000686806,0.000686794,0.000686782,0.000686769,0.000686756,0.000686743,0.00068673,0.000686717,0.000686703,0.00068669,0.000686676,0.000686662,0.000686648,0.000686634,0.00068662,0.000686606,0.000686591,0.000686576,0.000775984,0.000775962,0.00077594,0.000775919,0.000775897,0.000775875,0.000775853,0.000775832,0.00077581,0.000775788,0.000775766,0.000775745,0.000775723,0.000775701,0.000775679,0.000775658,0.000775636,0.000775614,0.000775592,0.000775571,0.000775549,0.000775527,0.000775505,0.000775484,0.000775462,0.00077544,0.000775418,0.000775396,0.000775375,0.000775353,0.000775331,0.000775309,0.000775288,0.000775266,0.000775244,0.000775222,0.000775201,0.000775179,0.000775157,0.000775135,0.000775114,0.000775092,0.00077507,0.000775048,0.000775026,0.000775005,0.000774983,0.000774961,0.000774939,0.000774918,0.000774896,0.000774874,0.000872263,0.000872238,0.000872213,0.000872188,0.000872163,0.000872138,0.000872112,0.000872087,0.000872062,0.000872037,NA_real_),
    q_adjusted = c(0.013771794,0.005727643,0.00439791,0.003708892,0.003268321,0.002955252,0.002717958,0.002530052,0.002376459,0.002247856,0.002138121,0.002043046,0.001959631,0.001885672,0.001819506,0.001759854,0.001705711,0.00165628,0.001610913,0.001569083,0.001530351,0.001494355,0.001460786,0.001429382,0.001399922,0.001372211,0.001346084,0.001321396,0.001298018,0.001275839,0.001254759,0.001234692,0.001215557,0.001197286,0.001179815,0.001163087,0.001147051,0.001131661,0.001116875,0.001102653,0.001088961,0.001075768,0.001063042,0.001050758,0.00103889,0.001027415,0.001016313,0.001005563,0.000995147,0.000985048,0.00097525,0.00096574,0.000956502,0.000947524,0.000938795,0.000930302,0.000922036,0.000913987,0.000906145,0.000898501,0.000891047,0.000883776,0.000876681,0.000869753,0.000862987,0.000856377,0.000849916,0.000843599,0.000837421,0.000831377,0.000825462,0.000819672,0.000814001,0.000808447,0.000803005,0.000797671,0.000792443,0.000787315,0.000782286,0.000777353,0.000772511,0.000767759,0.000763093,0.000758512,0.000754012,0.000749591,0.000745247,0.000740978,0.000736781,0.000732655,0.000728597,0.000724606,0.00072068,0.000716817,0.00076301,0.000761325,0.000759665,0.00075803,0.00075642,0.000754834,0.000753272,0.000751732,0.000750215,0.000748719,0.000747244,0.00074579,0.000744357,0.000742942,0.000741548,0.000740171,0.000738814,0.000737474,0.000736152,0.000734846,0.000733558,0.000732286,0.00073103,0.000729789,0.000728564,0.000727354,0.000726158,0.000724977,0.00072381,0.000722657,0.000721517,0.000720391,0.000719277,0.000718176,0.000717088,0.000716011,0.000714947,0.000713894,0.000712853,0.000711823,0.000710804,0.000709796,0.000708799,0.000707812,0.000706835,0.000705869,0.000704912,0.000703965,0.000703027,0.000702099,0.00070118,0.00070027,0.000775984,0.000775962,0.00077594,0.000775919,0.000775897,0.000775875,0.000775853,0.000775832,0.00077581,0.000775788,0.000775766,0.000775745,0.000775723,0.000775701,0.000775679,0.000775658,0.000775636,0.000775614,0.000775592,0.000775571,0.000775549,0.000775527,0.000775505,0.000775484,0.000775462,0.00077544,0.000775418,0.000775396,0.000775375,0.000775353,0.000775331,0.000775309,0.000775288,0.000775266,0.000775244,0.000775222,0.000775201,0.000775179,0.000775157,0.000775135,0.000775114,0.000775092,0.00077507,0.000775048,0.000775026,0.000775005,0.000774983,0.000774961,0.000774939,0.000774918,0.000774896,0.000774874,0.000872263,0.000872238,0.000872213,0.000872188,0.000872163,0.000872138,0.000872112,0.000872087,0.000872062,0.000872037,NA_real_),
    s_genmort  = c(1,0.999456646,0.998913598,0.998370858,0.997828424,0.997286297,0.996744476,0.996202961,0.995661752,0.995120849,0.994580252,0.99403996,0.993499974,0.992960292,0.992420916,0.991881844,0.991343077,0.990804614,0.990266455,0.9897286,0.98919105,0.988653802,0.988116859,0.987580219,0.987043881,0.986507847,0.985972116,0.985436687,0.984901561,0.984366737,0.983832214,0.983297994,0.982764076,0.982230459,0.981697144,0.981164129,0.980631416,0.980099004,0.979566892,0.979035081,0.97850357,0.977972359,0.977441448,0.976850118,0.976259159,0.975668572,0.975078355,0.974488509,0.973899034,0.973309928,0.972721193,0.972132827,0.971544831,0.970957204,0.970369946,0.969783057,0.969196536,0.968610384,0.9680246,0.967439183,0.966854134,0.966269452,0.965685138,0.96510119,0.964517609,0.963934395,0.963351546,0.962769063,0.962186946,0.961605195,0.961023808,0.960442787,0.95986213,0.959281838,0.95870191,0.958122346,0.957543146,0.956964309,0.956385836,0.955807726,0.955229978,0.954652593,0.954075571,0.95349891,0.952922612,0.952346675,0.9517711,0.951195885,0.950621032,0.95004654,0.949472408,0.948898636,0.948325224,0.947752172,0.947179479,0.946529964,0.945880911,0.945232318,0.944584185,0.943936513,0.943289301,0.942642548,0.941996255,0.94135042,0.940705044,0.940060126,0.939415666,0.938771663,0.938128118,0.937485029,0.936842397,0.936200222,0.935558502,0.934917237,0.934276428,0.933636073,0.932996173,0.932356728,0.931717736,0.931079197,0.930441112,0.92980348,0.9291663,0.928529572,0.927893297,0.927257472,0.926622099,0.925987177,0.925352706,0.924718684,0.924085113,0.923451991,0.922819318,0.922187094,0.921555319,0.920923992,0.920293113,0.919662681,0.919032697,0.918403159,0.917774069,0.917145424,0.916517225,0.915889473,0.915262165,0.914635302,0.914008884,0.913301397,0.912594477,0.911888124,0.911182337,0.910477117,0.909772462,0.909068373,0.908364848,0.907661888,0.906959491,0.906257657,0.905556387,0.904855679,0.904155533,0.903455948,0.902756924,0.902058461,0.901360558,0.900663214,0.89996643,0.899270204,0.898574537,0.897879427,0.897184875,0.896490879,0.89579744,0.895104556,0.894412228,0.893720456,0.893029237,0.892338573,0.891648462,0.890958905,0.8902699,0.889581448,0.888893547,0.888206197,0.887519399,0.886833151,0.886147453,0.885462304,0.884777704,0.884093653,0.88341015,0.882727195,0.882044787,0.881362926,0.880681612,0.880000843,0.879320619,0.878640941,0.877961807,0.877197821,0.876434522,0.875671909,0.874909982,0.87414874,0.873388182,0.872628308,0.871869117,0.871110609,0.870352783),
    s_adjusted = c(1,0.986228206,0.980579443,0.976266943,0.972646074,0.969467155,0.966602135,0.963974951,0.961536044,0.959250993,0.957094735,0.955048351,0.953097143,0.951229425,0.949435718,0.947708214,0.946040386,0.944426714,0.942862479,0.94134361,0.939866564,0.938428238,0.937025893,0.935657099,0.934319687,0.933011713,0.931731423,0.930477234,0.929247706,0.928041526,0.926857495,0.925694512,0.924551565,0.923427719,0.922322112,0.921233943,0.920162467,0.919106994,0.918066876,0.91704151,0.916030331,0.91503281,0.914048447,0.913076775,0.912117353,0.911169763,0.910233613,0.909308531,0.908394165,0.90749018,0.906596258,0.9057121,0.904837418,0.903971939,0.903115404,0.902267564,0.901428182,0.900597033,0.899773899,0.898958573,0.898150858,0.897350563,0.896557506,0.895771512,0.894992412,0.894220045,0.893454256,0.892694895,0.891941818,0.891194887,0.890453968,0.889718932,0.888989654,0.888266015,0.887547899,0.886835194,0.886127791,0.885425586,0.884728477,0.884036366,0.883349158,0.882666761,0.881989085,0.881316046,0.880647557,0.879983538,0.879323911,0.878668597,0.878017524,0.877370617,0.876727807,0.876089026,0.875454206,0.874823284,0.874196196,0.873529175,0.872864135,0.872201051,0.871539897,0.870880646,0.870223276,0.869567761,0.868914079,0.868262207,0.867612123,0.866963805,0.866317231,0.865672382,0.865029238,0.864387777,0.863747982,0.863109833,0.862473312,0.861838401,0.861205082,0.860573339,0.859943153,0.859314509,0.858687391,0.858061782,0.857437667,0.856815032,0.85619386,0.855574139,0.854955852,0.854338987,0.853723529,0.853109465,0.852496782,0.851885468,0.851275508,0.850666891,0.850059605,0.849453638,0.848848977,0.848245612,0.84764353,0.847042721,0.846443174,0.845844879,0.845247823,0.844651998,0.844057393,0.843463997,0.842871802,0.842280797,0.841690973,0.841037835,0.840385221,0.839733132,0.839081568,0.838430527,0.83778001,0.837130016,0.836480544,0.835831594,0.835183166,0.834535259,0.833887873,0.833241007,0.832594661,0.831948835,0.831303527,0.830658738,0.830014468,0.829370715,0.82872748,0.828084761,0.827442559,0.826800873,0.826159702,0.825519047,0.824878907,0.824239281,0.823600168,0.82296157,0.822323484,0.821685911,0.82104885,0.820412301,0.819776263,0.819140736,0.81850572,0.817871214,0.817237218,0.81660373,0.815970752,0.815338282,0.81470632,0.814074866,0.813443918,0.812813478,0.812183543,0.811554115,0.810925192,0.810296774,0.809668861,0.809041452,0.808414547,0.807709396,0.807004881,0.806301001,0.805597755,0.804895142,0.804193163,0.803491816,0.802791101,0.802091017,0.801391564),
    prop_male.genmort  = c(0.55,0.549945861,0.54989172,0.549837578,0.549783435,0.549729291,0.549675146,0.549620999,0.549566852,0.549512703,0.549458553,0.549404401,0.549350249,0.549296096,0.549241941,0.549187785,0.549133628,0.54907947,0.54902531,0.54897115,0.548916988,0.548862825,0.548808661,0.548754496,0.54870033,0.548646162,0.548591994,0.548537824,0.548483653,0.548429481,0.548375308,0.548321133,0.548266958,0.548212781,0.548158603,0.548104424,0.548050244,0.547996063,0.547941881,0.547887697,0.547833513,0.547779327,0.54772514,0.547666264,0.547607386,0.547548507,0.547489627,0.547430745,0.547371863,0.547312978,0.547254093,0.547195206,0.547136318,0.547077428,0.547018537,0.546959645,0.546900752,0.546841857,0.546782961,0.546724064,0.546665165,0.546606265,0.546547364,0.546488461,0.546429557,0.546370652,0.546311745,0.546252838,0.546193929,0.546135018,0.546076106,0.546017193,0.545958279,0.545899364,0.545840447,0.545781529,0.545722609,0.545663689,0.545604767,0.545545843,0.545486919,0.545427993,0.545369066,0.545310138,0.545251208,0.545192277,0.545133345,0.545074412,0.545015477,0.544956541,0.544897604,0.544838665,0.544779726,0.544720785,0.544661843,0.544597431,0.544533017,0.544468602,0.544404186,0.544339768,0.544275348,0.544210927,0.544146505,0.544082081,0.544017656,0.543953229,0.543888801,0.543824371,0.54375994,0.543695507,0.543631073,0.543566637,0.5435022,0.543437762,0.543373322,0.54330888,0.543244437,0.543179993,0.543115547,0.5430511,0.542986651,0.542922201,0.54285775,0.542793297,0.542728842,0.542664387,0.542599929,0.542535471,0.542471011,0.542406549,0.542342086,0.542277622,0.542213156,0.542148689,0.54208422,0.54201975,0.541955279,0.541890806,0.541826332,0.541761856,0.541697379,0.541632901,0.541568421,0.541503939,0.541439457,0.541374973,0.541310488,0.541236898,0.541163306,0.541089713,0.541016118,0.540942521,0.540868922,0.540795322,0.54072172,0.540648116,0.54057451,0.540500902,0.540427293,0.540353682,0.540280069,0.540206455,0.540132838,0.54005922,0.5399856,0.539911979,0.539838355,0.53976473,0.539691103,0.539617475,0.539543845,0.539470213,0.539396579,0.539322943,0.539249306,0.539175667,0.539102027,0.539028385,0.538954741,0.538881095,0.538807447,0.538733798,0.538660148,0.538586495,0.538512841,0.538439185,0.538365528,0.538291868,0.538218208,0.538144545,0.538070881,0.537997215,0.537923548,0.537849878,0.537776208,0.537702535,0.537628861,0.537555185,0.537481508,0.537402314,0.537323119,0.537243921,0.537164722,0.537085521,0.537006318,0.536927113,0.536847906,0.536768697,0.536689487 ),
    prop_male.adjusted = c(0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.55,0.549977485,0.549954043,0.549929689,0.549904436,0.549878299,0.54985129,0.549823422,0.549794709,0.549765163,0.549734795,0.549703618,0.549671642,0.54963888,0.549605342,0.549571039,0.549535981,0.549500179,0.549463643,0.549426381,0.549388405,0.549349723,0.549310345,0.549270279,0.549229533,0.549188118,0.549146041,0.54910331,0.549059933,0.549015919,0.548971275,0.548926008,0.548880126,0.548833636,0.548786546,0.548738862,0.548690591,0.548641739,0.548592315,0.548542323,0.548491771,0.548440664,0.548389008,0.54833681,0.548284076,0.548230811,0.548177021,0.548122711,0.548067887,0.548012555,0.547956719,0.547900385,0.547843558,0.547770141,0.547696721,0.5476233,0.547549877,0.547476451,0.547403023,0.547329594,0.547256162,0.547182728,0.547109292,0.547035855,0.546962415,0.546888973,0.546815529,0.546742083,0.546668635,0.546595185,0.546521732,0.546448278,0.546374822,0.546301364,0.546227904,0.546154441,0.546080977,0.546007511,0.545934043,0.545860572,0.5457871,0.545713626,0.54564015,0.545566672,0.545493191,0.545419709,0.545346225,0.545272739,0.545199251,0.545125761,0.545052269,0.544978775,0.544905279,0.544831781,0.544758281,0.544684779,0.544611275,0.544537769,0.544464262,0.544390752,0.544317241,0.544243727,0.544170212,0.544096695,0.544023175,0.543944151,0.543865125,0.543786097,0.543707066,0.543628033,0.543548998,0.543469961,0.543390922,0.54331188,0.543232837)
  )
  
  testthat::expect_equal(adjusted, target, tolerance = 1e-6)
  
  # Restore or remove i and p
  if (!is.null(old_i)) assign("i", old_i, envir = globalenv()) else rm("i", envir = globalenv())
  if (!is.null(old_p)) assign("p", old_p, envir = globalenv()) else rm("p", envir = globalenv())
  
})


testthat::test_that("the wrapper function works with individual patient characteristics", {
  
  i <- list(R_table_mort_lifeTable = data.frame(
    V1 = 0:100,
    V2 = c(0.004233,0.000229,0.000127,0.000102,0.000086,0.000074,0.000085,0.000067,0.000069,0.00006,0.000078,0.000077,0.000102,0.000116,0.000129,0.000172,0.000205,0.000311,0.000402,0.000454,0.000525,0.000507,0.000498,0.000524,0.000556,0.000601,0.000607,0.00063,0.000681,0.000728,0.000771,0.000835,0.000859,0.000958,0.00099,0.0011,0.001155,0.001352,0.001317,0.001458,0.001607,0.001701,0.00185,0.002017,0.002212,0.00247,0.00265,0.002746,0.002964,0.003303,0.003584,0.003828,0.004084,0.004411,0.004731,0.005059,0.005609,0.006078,0.006718,0.007265,0.007944,0.008673,0.009648,0.010608,0.011234,0.012538,0.01395,0.014892,0.016483,0.018111,0.019424,0.021013,0.023044,0.026136,0.029196,0.03283,0.03638,0.040958,0.046233,0.051555,0.057764,0.064015,0.071677,0.080609,0.09128,0.10247,0.11665,0.130842,0.147828,0.16745,0.178043,0.201824,0.223296,0.250961,0.27869,0.310738,0.339857,0.372549,0.403135,0.453901,0.485254),
    V3 = c(0.003503,0.000214,0.000114,0.000095,0.000064,0.000074,0.000071,0.000055,0.000058,0.000051,0.000066,0.000055,0.000057,0.000087,0.000096,0.000113,0.000131,0.000158,0.000218,0.000212,0.000187,0.000211,0.000245,0.000215,0.000223,0.00026,0.000257,0.00031,0.000314,0.000338,0.000387,0.000394,0.000468,0.000493,0.000585,0.000592,0.000678,0.000761,0.000792,0.000868,0.000924,0.001004,0.001102,0.001239,0.001345,0.001484,0.001625,0.001744,0.001966,0.002051,0.002234,0.002452,0.002581,0.002764,0.002964,0.003283,0.003637,0.003928,0.004367,0.004707,0.005247,0.005636,0.006451,0.006818,0.007379,0.008113,0.00877,0.009554,0.010602,0.011458,0.012895,0.013637,0.015499,0.017289,0.019688,0.021766,0.024397,0.027918,0.03142,0.035713,0.039611,0.045127,0.050197,0.057155,0.065142,0.073748,0.08452,0.095918,0.107798,0.121609,0.136466,0.153438,0.171026,0.189568,0.20787,0.230227,0.253171,0.277939,0.299649,0.31991,0.350742)
  ))
  p <- list(basic = list(cl_y = 1/52))
  
  # Create a survivor function to use
  t_os <- (0:208) * p$basic$cl_y
  s_os <- exp(-0.05 * t_os ^ 0.5)
  
  # Test using individual ages and sexes
  ages <- c(57.5, 72.6, 67.9, 71.1)
  sexes <- c("M", "M", "F", "F")
  testthat::expect_warning(
    adjusted <- adjust_survival(sex = sexes, age = ages, survivor = data.frame(t = t_os, s = s_os), .i = i, .p = p, .warn = TRUE)
  )
  
  # The values below were calculated in Excel
  target <- data.frame(
    t = t_os,
    q_genmort  = c(0.000253523,0.000253523,0.000253523,0.000253523,0.000253523,0.000253523,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000258611,0.000273844,0.000273844,0.000273844,0.000273844,0.000273844,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000276941,0.000286022,0.000286022,0.000286021,0.00028602,0.000286019,0.000286018,0.000286016,0.000286014,0.000286012,0.000286009,0.000286007,0.000290164,0.000290161,0.000290158,0.000290155,0.000290151,0.000290147,0.000290143,0.000290139,0.000290134,0.00029013,0.000290125,0.00029012,0.000290115,0.00029011,0.000290104,0.000305203,0.000305192,0.000305181,0.00030517,0.000305158,0.000307796,0.000307785,0.000307773,0.000307761,0.000307749,0.000307736,0.000307724,0.000307711,0.000307699,0.000307686,0.000307673,0.00030766,0.000307646,0.000307633,0.000307619,0.000307606,0.000307592,0.000307578,0.000307564,0.00030755,0.000307536,0.000316282,0.000316268,0.000316254,0.00031624,0.000316225,0.000316211,0.000316196,0.000316182,0.000316167,0.000316152,0.000316137,0.00032313,0.000323116,0.000323101,0.000323086,0.000323071,0.000323056,0.000323041,0.000323026,0.00032301,0.000322995,0.000322979,0.000322964,0.000322948,0.000322932,0.000322916,0.000340747,0.000340723,0.000340698,0.000340673,0.000340648,0.000343924,0.000343899,0.000343874,0.000343849,0.000343824,0.000343799,0.000343774,0.000343748,0.000343723,0.000343698,0.000343672,0.000343647,0.000343621,0.000343595,0.000343569,0.000343543,0.000343517,0.000343491,0.000343465,0.000343439,0.000343413,0.000355186,0.00035516,0.000355134,0.000355108,0.000355081,0.000355055,0.000355029,0.000355002,0.000354976,0.000354949,0.000354922,0.000358535,0.000358509,0.000358483,0.000358456,0.00035843,0.000358403,0.000358376,0.00035835,0.000358323,0.000358296,0.000358269,0.000358242,0.000358215,0.000358188,0.000358161,0.000375405,0.000375367,0.00037533,0.000375293,0.000375255,0.000378785,0.000378748,0.000378711,0.000378674,0.000378637,0.0003786,0.000378563,0.000378525,0.000378488,0.000378451,0.000378414,0.000378376,0.000378339,0.000378302,0.000378264,0.000378227,0.000378189,0.000378152,0.000378114,0.000378077,0.000378039,0.000388252,0.000388214,0.000388176,0.000388138,0.0003881,NA_real_ ),
    q_adjusted = c(0.006909769,0.002867934,0.002201378,0.001856169,0.001635498,0.001478719,0.001359904,0.001265827,0.001188936,0.00112456,0.001069632,0.001022045,0.000980296,0.000943281,0.000910167,0.000880314,0.00085322,0.000828483,0.000805781,0.000784849,0.000765469,0.000747457,0.00073066,0.000714947,0.000700206,0.000686341,0.000673269,0.000660916,0.00064922,0.000638123,0.000627577,0.000617536,0.000607963,0.000598822,0.000590082,0.000581713,0.00057369,0.000565991,0.000558593,0.000551479,0.000544629,0.000538029,0.000531662,0.000525517,0.00051958,0.00051384,0.000508507,0.000504473,0.000500565,0.000496777,0.000493101,0.000489532,0.000486066,0.000482698,0.000479422,0.000476236,0.000473134,0.000470113,0.00046717,0.000464302,0.000461505,0.000458776,0.000456112,0.000453512,0.000450972,0.000448491,0.000446065,0.000443693,0.000441374,0.000439104,0.000436883,0.000434708,0.000432578,0.000445596,0.000443548,0.000441541,0.000439573,0.000437643,0.00043575,0.000433891,0.000432068,0.000430277,0.000428519,0.000426793,0.000425096,0.000423429,0.000421791,0.000420181,0.000418597,0.00041704,0.000415508,0.000414001,0.000412519,0.000411059,0.000409623,0.000408209,0.000406816,0.000405445,0.000404094,0.000402764,0.000401453,0.000400161,0.000398887,0.000397632,0.000396394,0.000395174,0.000393971,0.000393037,0.000392253,0.00039148,0.000390717,0.000389963,0.00038922,0.000388486,0.000387761,0.000387046,0.00038634,0.000385642,0.000384953,0.000384272,0.0003836,0.000382935,0.000382279,0.00038163,0.000380989,0.000398202,0.000397568,0.00039694,0.00039632,0.000395707,0.0003951,0.000394499,0.000393906,0.000393318,0.000392737,0.000392161,0.000391592,0.000391029,0.000390471,0.000389919,0.000389372,0.000388831,0.000388295,0.000387765,0.000387239,0.000386719,0.000386203,0.000385693,0.000385187,0.000384686,0.00038419,0.000395497,0.000395011,0.000394529,0.000394051,0.000393577,0.000393108,0.000392642,0.000392181,0.000391724,0.00039127,0.000390821,0.000390375,0.000389933,0.000389495,0.00038906,0.000388629,0.000388202,0.000387778,0.000387357,0.00038694,0.000386526,0.000386138,0.000385922,0.000385708,0.000385496,0.000385285,0.000402347,0.000402129,0.000401913,0.000401698,0.000401485,0.000401273,0.000401062,0.000400853,0.000400646,0.000400439,0.000400234,0.00040003,0.000399828,0.000399626,0.000399426,0.000399228,0.00039903,0.000398834,0.000398638,0.000398444,0.000398251,0.00039806,0.000397869,0.000397679,0.000397491,0.000397303,0.000407367,0.000407182,0.000406998,0.000406814,0.000406632,NA_real_),
    s_genmort  = c(1,0.999746477,0.999493034,0.999239671,0.998986387,0.998733183,0.998480058,0.99822193,0.997963882,0.997705917,0.997448032,0.997190229,0.996932508,0.996674867,0.996417308,0.99615983,0.995902433,0.995645118,0.995387883,0.99513073,0.994873657,0.994616666,0.994344666,0.994072761,0.993800951,0.993529237,0.993257618,0.992983007,0.992708492,0.992434074,0.992159751,0.991885524,0.991611393,0.991337357,0.991063418,0.990789574,0.990515825,0.990242173,0.989968616,0.989695154,0.989421788,0.989148518,0.988875343,0.988602263,0.988329279,0.98805639,0.987783596,0.987510897,0.987229324,0.986947852,0.986666479,0.986385206,0.986104034,0.985822962,0.985541989,0.985261117,0.984980344,0.984699672,0.984419099,0.984134514,0.983850031,0.983565649,0.983281369,0.98299719,0.982713113,0.982429136,0.982145261,0.981861488,0.981577815,0.981294244,0.981010773,0.980727404,0.980444136,0.980160968,0.979863313,0.979565774,0.979268352,0.978971047,0.978673858,0.978374163,0.978074585,0.977775125,0.977475781,0.977176554,0.976877444,0.976578451,0.976279574,0.975980814,0.975682171,0.975383644,0.975085234,0.974786941,0.974488763,0.974190702,0.973892758,0.973594929,0.973297217,0.972999621,0.972702141,0.972404777,0.972099025,0.971793393,0.971487883,0.971182494,0.970877227,0.97057208,0.970267055,0.96996215,0.969657367,0.969352704,0.969048162,0.96873691,0.968425781,0.968114777,0.967803896,0.967493139,0.967182506,0.966871996,0.96656161,0.966251347,0.965941208,0.965631192,0.9653213,0.965011531,0.964701885,0.964392363,0.964066087,0.963739954,0.963413965,0.96308812,0.962762417,0.962433625,0.962104977,0.961776472,0.961448112,0.961119896,0.960791823,0.960463894,0.960136108,0.959808467,0.959480968,0.959153613,0.958826402,0.958499333,0.958172408,0.957845626,0.957518987,0.957192491,0.956866138,0.956539928,0.95621386,0.955887935,0.955550928,0.955214072,0.954877366,0.954540811,0.954204406,0.953868151,0.953532046,0.953196092,0.952860287,0.952524632,0.952189128,0.951850287,0.951511598,0.95117306,0.950834674,0.950496439,0.950158355,0.949820422,0.94948264,0.949145009,0.948807529,0.9484702,0.948133022,0.947795994,0.947459116,0.94712239,0.94676981,0.946417402,0.946065166,0.945713102,0.945361209,0.945006042,0.944651048,0.944296226,0.943941576,0.943587099,0.943232794,0.942878661,0.9425247,0.942170911,0.941817293,0.941463847,0.941110573,0.940757471,0.940404539,0.94005178,0.939699191,0.939346773,0.938994526,0.938642451,0.938290546,0.937938811,0.937577695,0.937216757,0.936855997,0.936495415,0.936135011 ),
    s_adjusted = c(1,0.993090231,0.990242113,0.988062216,0.986228206,0.984615232,0.983159262,0.98182226,0.980579443,0.979413596,0.978312187,0.977265752,0.976266943,0.975309912,0.974389921,0.973503063,0.972646074,0.971816193,0.97101106,0.970228638,0.969467155,0.968725058,0.968000978,0.967293698,0.966602135,0.965925314,0.96526236,0.964612479,0.963974951,0.963349119,0.962734384,0.962130195,0.961536044,0.960951466,0.960376026,0.959809326,0.959250993,0.95870068,0.958158064,0.957622843,0.957094735,0.956573473,0.956058809,0.955550509,0.955048351,0.954552127,0.95406164,0.953576493,0.953095439,0.952618353,0.952145114,0.951675611,0.951209735,0.950747384,0.950288461,0.949832871,0.949380527,0.948931343,0.948485237,0.948042133,0.947601955,0.947164633,0.946730097,0.946298282,0.945869124,0.945442563,0.945018541,0.944597002,0.94417789,0.943761155,0.943346746,0.942934614,0.942524713,0.942116998,0.941697195,0.941279506,0.940863893,0.940450314,0.940038733,0.939629111,0.939221414,0.938815607,0.938411656,0.938009528,0.937609193,0.937210618,0.936813776,0.936418636,0.936025172,0.935633354,0.935243158,0.934854557,0.934467526,0.93408204,0.933698077,0.933315613,0.932934625,0.932555092,0.932176992,0.931800305,0.931425009,0.931051086,0.930678516,0.93030728,0.929937361,0.929568739,0.929201397,0.928835319,0.928470252,0.928106057,0.927742722,0.927380238,0.927018594,0.92665778,0.926297786,0.925938603,0.925580223,0.925222634,0.924865829,0.924509799,0.924154536,0.92380003,0.923446275,0.923093261,0.92274098,0.922389426,0.922022129,0.921655563,0.921289721,0.920924595,0.920560179,0.920196466,0.919833449,0.919471121,0.919109477,0.918748509,0.918388211,0.918028577,0.917669602,0.917311279,0.916953602,0.916596565,0.916240164,0.915884393,0.915529245,0.915174716,0.914820801,0.914467494,0.914114791,0.913762685,0.913411174,0.91306025,0.912699137,0.912338611,0.911978668,0.911619302,0.911260509,0.910902286,0.910544627,0.910187529,0.909830987,0.909474997,0.909119555,0.908764657,0.908410299,0.908056478,0.907703189,0.907350429,0.906998194,0.90664648,0.906295284,0.905944603,0.905594432,0.905244747,0.904895393,0.904546367,0.904197668,0.903849294,0.903485634,0.903122316,0.902759339,0.902396703,0.902034404,0.901672442,0.901310815,0.900949522,0.90058856,0.900227929,0.899867627,0.899507653,0.899148005,0.898788682,0.898429682,0.898071004,0.897712647,0.897354609,0.896996889,0.896639486,0.896282398,0.895925624,0.895569163,0.895213014,0.894857175,0.894501645,0.894137255,0.893773178,0.893409415,0.893045963,0.892682822),
    prop_male.genmort  = c(0.5,0.499985389,0.499970785,0.499956187,0.499941596,0.499927011,0.499912432,0.499900406,0.499888386,0.499876372,0.499864366,0.499852366,0.499840373,0.499828386,0.499816406,0.499804432,0.499792466,0.499780505,0.499768552,0.499756605,0.499744665,0.499732731,0.499713212,0.499693702,0.499674202,0.499654711,0.499635229,0.499614201,0.499593182,0.499572172,0.499551171,0.499530178,0.499509194,0.499488219,0.499467253,0.499446295,0.499425347,0.499404407,0.499383476,0.499362553,0.49934164,0.499320735,0.499299839,0.499278952,0.499258074,0.499237204,0.499216343,0.499195492,0.499179184,0.499162884,0.499146593,0.499130311,0.499114037,0.499097771,0.499081514,0.499065265,0.499049025,0.499032793,0.49901657,0.49900244,0.498988318,0.498974206,0.498960101,0.498946006,0.498931919,0.49891784,0.498903771,0.498889709,0.498875657,0.498861613,0.498847578,0.498833551,0.498819533,0.498805524,0.498784061,0.498762609,0.49874117,0.498719742,0.498698326,0.498675578,0.498652841,0.498630115,0.4986074,0.498584697,0.498562005,0.498539324,0.498516654,0.498493995,0.498471347,0.49844871,0.498426085,0.498403471,0.498380868,0.498358276,0.498335695,0.498313125,0.498290567,0.49826802,0.498245483,0.498222958,0.498204803,0.498186659,0.498168525,0.498150402,0.498132289,0.498114188,0.498096097,0.498078017,0.498059947,0.498041888,0.49802384,0.498009315,0.4979948,0.497980297,0.497965805,0.497951324,0.497936853,0.497922394,0.497907946,0.497893509,0.497879084,0.497864669,0.497850265,0.497835872,0.497821491,0.49780712,0.49778397,0.497760834,0.497737714,0.497714609,0.497691519,0.497666757,0.497642009,0.497617276,0.497592558,0.497567853,0.497543164,0.497518488,0.497493828,0.497469181,0.497444549,0.497419932,0.497395329,0.49737074,0.497346166,0.497321607,0.497297062,0.497272531,0.497248015,0.497223513,0.497199025,0.497174553,0.497155934,0.497137329,0.497118738,0.497100161,0.497081598,0.497063048,0.497044513,0.497025991,0.497007483,0.496988988,0.496970508,0.496953861,0.496937228,0.496920609,0.496904005,0.496887414,0.496870837,0.496854275,0.496837726,0.496821191,0.496804671,0.496788164,0.496771672,0.496755193,0.496738729,0.496722278,0.496697335,0.49667241,0.496647503,0.496622616,0.496597747,0.49657106,0.496544392,0.496517741,0.496491108,0.496464493,0.496437895,0.496411315,0.496384753,0.496358208,0.496331681,0.496305172,0.49627868,0.496252206,0.49622575,0.496199311,0.49617289,0.496146487,0.496120102,0.496093734,0.496067384,0.496041051,0.49601979,0.495998546,0.495977319,0.495956109,0.495934915 ),
    prop_male.adjusted = c(0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.5,0.499999889,0.499999106,0.499997671,0.499995604,0.499992925,0.49998965,0.499985798,0.499981384,0.499976424,0.499970933,0.499964925,0.499958414,0.499951412,0.499943933,0.499935988,0.499927588,0.499918745,0.499909469,0.49989977,0.499889658,0.499879143,0.499868233,0.499856937,0.499845264,0.499833222,0.499820819,0.499808061,0.499787399,0.499766398,0.499745066,0.499723408,0.499701433,0.499679145,0.499656551,0.499633656,0.499610467,0.499586989,0.499563227,0.499539186,0.499514872,0.499490288,0.499465441,0.499440334,0.499414973,0.499389361,0.499363502,0.499337401,0.499311062,0.499284489,0.499257686,0.499230655,0.499203401,0.499175928,0.499148239,0.499120337,0.499092225,0.499063907,0.499035385,0.499006664,0.498977745,0.498948631,0.498919453,0.498890279,0.49886111,0.498831945,0.498802785,0.498773629,0.498744477,0.498715331,0.498686189,0.498657051,0.498627919,0.498598791,0.498569669,0.498540551,0.498511438,0.49848233,0.498453228,0.49842413,0.498386082,0.498348042,0.49831001,0.498271986,0.49823397,0.498195962,0.498157962,0.498119971,0.498081987,0.498044013,0.498006046,0.497968088,0.497930139,0.497892198,0.497854265,0.497816342,0.497778427,0.49774052,0.497702623,0.497664734,0.497626854,0.497588984,0.497551122,0.497513269,0.497475425,0.49743759,0.497405636,0.497373691,0.497341754,0.497309826,0.497277907,0.497245997,0.497214095,0.497182203,0.497150319,0.497118445,0.497086579,0.497054723,0.497022875,0.496991037,0.496959208,0.496927388,0.496895577,0.496863775,0.496831983,0.496800199,0.496768425,0.496736672,0.496705024,0.496673479,0.496642038,0.496610699,0.496570764,0.496530934,0.496491207,0.496451584,0.496412063,0.496372645,0.496333327,0.49629411,0.496254993,0.496215975,0.496177056,0.496138235,0.496099511,0.496060884,0.496022352,0.495983917,0.495945576,0.49590733,0.495869178,0.495831119,0.495793152,0.495755278,0.495717495,0.495679803,0.495642202,0.495604691,0.49557235,0.495540099,0.495507934,0.495475858,0.495443868)
  )
  
  testthat::expect_equal(adjusted, target, tolerance = 1e-6)
  
})