% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcda.R
\name{mcda}
\alias{mcda}
\title{Multi criteria decision analysis}
\usage{
mcda(x, sample, strategy, criteria, criteria_min = NULL,
  criteria_max = NULL, optimal = NULL, weights, score_min = 0,
  score_max = 100)
}
\arguments{
\item{x}{A \code{data.frame} or \code{data.table} of simulation output 
characterizing the probability distribution of model outcomes.}

\item{sample}{Character name of column from \code{x} denoting a randomly sampled 
parameter set from the PSA.}

\item{strategy}{Character name of column from \code{x} denoting treatment strategy.}

\item{criteria}{A vector of character names of columns from \code{x} denoting
the criteria to use in the MCDA.}

\item{criteria_min}{A vector of minimum values for each criterion. If \code{NULL}, 
then the minimum value is computed automatically.}

\item{criteria_max}{A vector of maximum values for each criterion. If \code{NULL}, 
then the maximum value is computed automatically.}

\item{optimal}{A character vector denoting whether the optimal value of each criteria
is \code{"low"} or \code{"high"}. If an element is \code{"low"}, then 
lower performance on that criterion is better, and,
if an element is \code{"high"}, then higher performance on that criterion is better. 
Must be specified if either \code{criteria_min} or 
\code{criteria_min} is \code{NULL}.}

\item{weights}{Weights to apply to each criteria. Internally normalized to
sum to 1.}

\item{score_min}{Minimum of total value score. Default is 0.}

\item{score_max}{Maximum of total value score. Default is 100.}
}
\description{
Conduct a multi criteria decision analysis (MCDA) and compute scores for
competing treatment strategies using output from a probabilistic sensitivity 
analysis (PSA).
}
\examples{
n_samples <- 5
strategies <- c("Strategy 1", "Strategy 2")
outcome1 <- c(rnorm(n_samples, mean = 10, sd = 5),
              rnorm(n_samples, mean = 8, sd = 4))
outcome2 <- c(rnorm(n_samples, mean = 1500, sd = 90),
              rnorm(n_samples, mean = 1000, sd = 100))
outcomes <- data.frame(sample = rep(1:n_samples, length(strategies)),
                       strategy_id = rep(strategies, each = n_samples),
                       criteria1 = outcome1,
                       criteria2 = outcome2)

# Performance matrix
performance_mat <- performance_matrix(outcomes, 
                                      strategy = "strategy_id", 
                                      criteria = c("criteria1", "criteria2"),
                                      rownames = c("Criteria 1", "Criteria 2"), 
                                      colnames = strategies)
print(performance_mat)
                                                                                                
# MCDA                        
weights <- c(.7, .3)
mcda <- mcda(outcomes, sample = "sample", strategy = "strategy_id",
             criteria = c("criteria1", "criteria2"),
             weights = weights,
             optimal = c("low", "high"))
names(mcda)

# Scores on common scale
print(mcda$scores)

# "Total value"
print(mcda$total_value)

# "Total value" decomposed by criteria
print(mcda$weighted_scores)

# Probability of ranking
print(mcda$prob_rank)
}
\seealso{
\code{\link{performance_matrix}}, \code{\link{lpvf_plot_data}}
}
